/** \file main.c
 * \brief  
 *
 * \details   
 *
 * \author \b ARV
 * \date	10  2021 .
 * \copyright 2021  ARV. All rights reserved.
 *
 */

#include "pumprelay.h"


static state_t state;
static state_t old_state;
static uint8_t delay;
static volatile uint8_t timeout;
static volatile uint8_t counter;

static EEMEM state_t e_state;
static EEMEM state_t e_delay;

///  
static void save(void){
	eeprom_update_byte(&e_delay, delay);
	eeprom_update_byte(&e_state, state);
}

///   
static bool power_lost(void){
	return !(PINB & POWER_PIN);
}

///  
static void pump(bool mode){
	if(mode){
		PORTB |= PUMP_PIN;
	} else {
		PORTB &= ~PUMP_PIN;
	}
}

///  
static void led(bool mode){
	if(!mode){
		PORTB |= LED_PIN;
	} else {
		PORTB &= ~LED_PIN;
	}
}

///       
static level_t get_level(uint16_t result){
	if((result >= HI_1) && (result <= HI_2)) return LEVEL_HI;
	else if((result >= LO_1) && (result <= LO_2)) return LEVEL_LO;
	else return LEVEL_UNKNOWN;
}

///     
static bool water_low(void){
	return !(PINB & WATER_PIN);
}

#define HALF_SEC	18
#define ONE_MIN		120

///    
ISR(TIM0_OVF_vect){
	static uint8_t precnt = HALF_SEC;
	static uint16_t min;

	if(timeout && (min == 0)){
		//       
		min = ONE_MIN;
	}

	//  
	if(!--precnt){
		precnt = HALF_SEC;
		counter++;				//     

		if(timeout){			//   
			if(!--min){			//    
				timeout--;		//    
				min = ONE_MIN;	//      
			}
		}

		if(!timeout) min = 0;	//   ,     
	}
}

///        
static uint16_t get_level_adc(void){
	uint16_t result = 0;
	for(uint8_t i=0; i<16; i++){
		ADCSRA |= _BV(ADSC);
		while(bit_is_set(ADCSRA, ADSC));
		result += ADC;
	}
	return result / 16;
}

#define abs(x)	((x) < 0 ? -(x) : (x))

//  
MAIN(){
	uint16_t adc = 0;
	level_t level;
	uint8_t wait;

	wdt_enable(WDTO_250MS);

	//  
	DDRB = LED_PIN | PUMP_PIN;
	PORTB = LEVEL_CTRL;

	//  
	state = eeprom_read_byte(&e_state);
	delay = eeprom_read_byte(&e_delay);

/*
	if(MCUSR & _BV(PORF)){
		MCUSR = 0;
		//   
		state = STATE_INIT;
	}
*/

	//  
	TCCR0B = TIMER_CLK_DIV_1024;
	TIMSK0 = _BV(TOIE0);

	//  
	ADCSRA = _BV(ADEN) | ADC_DIV_16;
	ADMUX = LEVEL_CTRL;

	sei();

	while(1){
		//  
		if(power_lost()){
			//   - 
			old_state = state;
			state = STATE_STOP;
		} else
			//     
			adc = get_level_adc();

		level = get_level(adc);

		//  
		switch(state){
		case STATE_INIT:
			//   
			state = STATE_OFF;
			delay = DEFAULT_WAIT;
			break;

		case STATE_ON:
			//  
			pump(true);
			led(true);
			if(water_low())
				// ,   
				state = STATE_NO_WATER;
			else if(level == LEVEL_HI){
				//     
				pump(false);
				led(false);
				state = STATE_DELAY;
				old_state = STATE_OFF;
				wait = counter;
			}
			break;

		case STATE_OFF:
			//  
			pump(false);
			led(false);
			if(level == LEVEL_LO)
				//     
				state = STATE_ON;
			break;

		case STATE_NO_WATER:
			//  
			pump(false);		//  
			led(counter & 1);	//  
			if(!water_low()){
				//   ,  
				timeout = delay;
				state = STATE_WAIT;
			}
			//    
			break;

		case STATE_DELAY:
			// 
			if(abs(counter - wait) > 10)
				state = old_state;
			break;

		case STATE_WAIT:
			//  
			led(counter & 1);	//  
			if(!timeout){
				//   ,  
				state = STATE_ON;
			}
			break;

		case STATE_STOP:
			//   
			state = old_state;
			save();
			return 0;

		default:
			//     EEPROM     
			state = STATE_INIT;
			continue;
		}
		wdt_reset();
	}
}

