#define F_CPU 9600000UL

#include <stdbool.h>

#include "pin_macros.h"

//----------------------------------------------------------------------------
//
//      -       ----     
//

#if defined(__GNUC__)                    //----    AVR-GCC

#  include <avr/io.h>
#  include <avr/interrupt.h>
#  include <avr/eeprom.h>

#  define SEI() sei()
#  define CLI() cli()

#  define EEPROM_SIZE (E2END+1)

#  define NOINLINE __attribute__((__noinline__))
#  define OS_MAIN  __attribute__((OS_main))

#elif defined(__IAR_SYSTEMS_ICC__)      //----    IAR

#  include <stdint.h>
#  include <ioavr.h>
#  include <intrinsics.h>

#  define SEI() __enable_interrupt()
#  define CLI() __disable_interrupt()

#  define EEMEM        __eeprom
#  define EEPROM_SIZE  64
#  define EEPE         EEWE
#  define EEMPE        EEMWE

#  define NOINLINE _Pragma("inline=never")
#  define OS_MAIN  __task

#else                                   //----    UNKNOWN (error)

#  error "Unknown compiler!"

#endif


//----------------------------------------------------------------------------
//
//      
//

#define DIV_ROUND(a,b) ((uint32_t)((float)(a)/(float)(b)+0.5))

//----------------------------------------------------------------------------
//
//       
//


#define BUTTON  B,1,L
#define LED     B,2,H
#define RX      B,3,H
#define RELAY   B,4,H


//----------------------------------------------------------------------------
//
//      ,   
//

#define ISR_FREQ 20000UL
#define OCR_VAL  (DIV_ROUND(F_CPU/8,ISR_FREQ) - 1)

#define US_TO_TICKS(us) DIV_ROUND(ISR_FREQ*(us), 1000000.)
#define MS_TO_TACKS(ms) DIV_ROUND(ISR_FREQ*(ms), 256000.)

enum {
    initial_delay_ms     = 200,
    initial_delay_tacks  = MS_TO_TACKS(initial_delay_ms),
      //   --      .
    led_store_ok_count   = 1,
    led_store_ok_ms      = 2000,
    led_store_ok_tacks   = MS_TO_TACKS(led_store_ok_ms),
      //   --    
    led_store_err_count  = 5,
    led_store_err_ms     = 200,
    led_store_err_tacks  = MS_TO_TACKS(led_store_err_ms),
      //    --  
    led_erase_count      = 30,
    led_erase_ms         = 50,
    led_erase_tacks      = MS_TO_TACKS(led_erase_ms),
      //   
    led_standby_ms       = 500,
    led_standby_tacks    = MS_TO_TACKS(led_standby_ms),
      //     
    relay_on_ms          = 500,
    relay_on_tacks       = MS_TO_TACKS(relay_on_ms)
};


//----------------------------------------------------------------------------
//
//      ,    
//


//
enum {
    key_bits      = 24,
    key_bytes     = 3,
    keys          = (EEPROM_SIZE-1)/key_bytes,
    key_data_size = keys*key_bytes
};

//------          
enum {
    bit_period_us     = 580,
    bit_period_min_us = (unsigned)(0.7*bit_period_us),
    bit_period_max_us = (unsigned)(1.4*bit_period_us),
    packet_end_us     = 2 * bit_period_max_us,
    //  , ,   packet_end_us
    inter_packet_gap_us   = 4500
};

enum {
    bit_period_min_ticks = US_TO_TICKS(bit_period_min_us),
    bit_period_max_ticks = US_TO_TICKS(bit_period_max_us),
    bit_high_max_ticks   = US_TO_TICKS(bit_period_max_us),
    packet_end_ticks     = US_TO_TICKS(packet_end_us)
};


//------        

typedef enum {
    rx_wait_start = 0, //   
    rx_count_hi,       //   HI,     rx_recover
    rx_count_lo,       //   LO,  ,     rx_wait_start
    rx_done,           //   ,   
    rx_recover         //  LO,     rx_wait_start
} rx_state_t;

typedef struct {
     // "public" -- volatile    ISR()   main()
    volatile rx_state_t state;
    volatile union {
        uint32_t dw;
        uint8_t  b[4];
    } data;
    volatile uint8_t    led_timer;    // 12.8 ms decrement, 3.28 s max time
    volatile uint8_t    relay_timer;  // 12.8 ms decrement, 3.28 s max time
     // "private" --    ISR
    uint8_t    tick;            // 50 us increment, 12.8 ms overflow
    uint8_t    base;
    uint8_t    width_hi;
    uint8_t    bits;
} receiver_t;


receiver_t receiver;

static void    packet_request() { receiver.state = rx_wait_start; }
static uint8_t packet_exist()   { return receiver.state == rx_done; }


//         --   
static void delay(uint8_t tacks)
{
    receiver.led_timer = tacks;
    while (receiver.led_timer) ;
}


// count         tacks   12.8 .
// count == 1 --        
//    127.
NOINLINE
static void blink_led(uint8_t tacks, uint8_t count)
{
    count = 2*count - 1;
    ON(LED);
    do {
        delay(tacks);
        TOGGLE(LED);
    } while(--count);
}

//----------------------------------------------------------------------------
//
//          .
//

//    
#define ADDR8(a) ((uint8_t)(unsigned)(a))

typedef struct {
    uint8_t index;
    uint8_t data[key_data_size];
} keymem_t;

EEMEM keymem_t keymem;

NOINLINE
static void EEWrite(uint8_t addr, uint8_t data)
{
    while (EECR & (1<<EEPE)) ;
    EECR = (0 << EEPM1) | (0 << EEPM0);
    EEAR = addr;
    EEDR = data;
    //       ,    
    CLI();
    EECR |= (1 << EEMPE);
    EECR |= (1 << EEPE);
    SEI();
} 


NOINLINE 
static uint8_t EERead(uint8_t addr)
{
    while (EECR & (1<<EEPE)) ;
    EEAR  = addr;
    EECR |= (1 << EERE);
    return EEDR;
} 


static uint8_t get_key_index()
{
    return EERead(ADDR8(&keymem.index));
}


static void erase_key_memory()
{
    uint8_t from  = ADDR8(&keymem);
    uint8_t up_to = ADDR8(&keymem) + sizeof(keymem);
    do EEWrite(from,0); while(++from < up_to);
}


static bool find_key(uint8_t b0, uint8_t b1, uint8_t b2)
{
    uint8_t index   = get_key_index();
    uint8_t from    = ADDR8(&keymem.data[0]);
    uint8_t up_to   = ADDR8(&keymem.data[index]);
    while (from < up_to) {
        if (EERead(from) == b0 && EERead(from+1) == b1 && EERead(from+2) == b2)
            return true;
        from += key_bytes;
    }
    return false;
}


static bool store_key(uint8_t b0, uint8_t b1, uint8_t b2)
{
    uint8_t index = get_key_index();
    if (index > key_data_size - key_bytes) {
        //     
        return false;
    } else {
        uint8_t ee_addr = ADDR8(&keymem.data[index]);
        EEWrite(ee_addr,b0);
        EEWrite(ee_addr+1,b1);
        EEWrite(ee_addr+2,b2);
        EEWrite(ADDR8(&keymem.index), index+key_bytes);
        return true;
    }
}


//----------------------------------------------------------------------------
//
//      Main routine
//

OS_MAIN void main()
{
    DDRB  = BITMASK(LED) | BITMASK(RELAY);
    PORTB = BITMASK(BUTTON); // LED off, RELAY off

    OCR0A  = OCR_VAL;
    TCCR0A = (1<<WGM01); // CTC
    TIMSK0 = (1<<OCIE0A);
    TCCR0B = (2<<CS00);  // clk/8

    SEI();

    delay(initial_delay_tacks);
    
    if (ACTIVE(BUTTON)) {
        erase_key_memory();
        blink_led(led_erase_tacks, led_erase_count);
    }

    for (;;) {
          //   .
        if (ACTIVE(BUTTON)) {
            OFF(LED);
        } else if (receiver.led_timer == 0) {
            receiver.led_timer = led_standby_tacks;
            TOGGLE(LED);
        }
          //  
        if (packet_exist()) {
            uint8_t b0 = receiver.data.b[0];
            uint8_t b1 = receiver.data.b[1];
            uint8_t b2 = receiver.data.b[2];
            bool key_found = find_key(b0,b1,b2);
            if (ACTIVE(BUTTON)) {
                if (key_found || store_key(b0,b1,b2)) {
                    blink_led(led_store_ok_tacks, led_store_ok_count);
                } else {
                    blink_led(led_store_err_tacks, led_store_err_count);
                }
            } else {
                  //     , ,    
                  //   .     ...
                if (key_found) {
                    receiver.relay_timer = relay_on_tacks;
                    ON(RELAY);
                }
            }
            packet_request(); //       receiver.data[]
        }
          //      for(;;)  .
        if (receiver.relay_timer == 0) {
            OFF(RELAY);
        }
    }
}

//----------------------------------------------------------------------------
//
//      Receiver ISR
//

#define DEC_TO_ZERO(type,var) \
    do { \
        type dec_to_zero_temp = var; \
        if (dec_to_zero_temp) var = dec_to_zero_temp - 1; \
    } while(0)


#if defined(__GNUC__)
ISR(TIM0_COMPA_vect)
#else
#pragma vector=TIM0_COMPA_vect
__interrupt void tim0_compa_isr(void)
#endif
{
    uint8_t tick = ++receiver.tick;
    if (tick == 0) {
        DEC_TO_ZERO(uint8_t, receiver.led_timer);
        DEC_TO_ZERO(uint8_t, receiver.relay_timer);
    }

    uint8_t width = tick - receiver.base;
    
    switch (receiver.state) {
    case rx_wait_start:
        if (ACTIVE(RX)) {
            receiver.bits = 0;
            receiver.state = rx_count_hi;
            receiver.base = tick;
        }
        break;

    case rx_count_hi:
        if (!ACTIVE(RX)) {
            receiver.width_hi = width;
            receiver.state = rx_count_lo;
            receiver.base = tick;
        } else if (width > bit_high_max_ticks) {
              //   ,   
            receiver.state = rx_recover;
        }
        break;

    case rx_count_lo:
        if (ACTIVE(RX)) {
            uint8_t width_hi = receiver.width_hi;
            uint8_t width_total = width_hi + width;
            if (width_total > bit_period_max_ticks || width_total < bit_period_min_ticks) {
                receiver.state = rx_wait_start;
            } else {
                uint32_t data = receiver.data.dw << 1;
                if (width < width_hi) data |= 1;
                receiver.data.dw = data;
                  //   1   ,     .
                if (receiver.bits < key_bits+1) ++receiver.bits;
                receiver.state = rx_count_hi;
                receiver.base = tick;
            }
        } else if (width > packet_end_ticks) {
            receiver.state = receiver.bits == key_bits ? rx_done : rx_wait_start;
        }
        break;

    case rx_done:
          //   ,     
        break;

    default:
          //   ,  rx_wait_start     rx_count_hi
        if (!ACTIVE(RX))
            receiver.state = rx_wait_start;
        break;
    }

} // ISR(TIM0_COMPA_vect)

