/* ----------------------------------------------------------------------- */
// Simple soldering station controller
// GP0 - Iron AD 
// GP1 - Iron Switch 
// GP2 - Iron Set 
// GP3 - On/Off 
// GP4 - Led 
// GP5 - Iron Heater 
/* ----------------------------------------------------------------------- */

#include <pic16regs.h>

/* ----------------------------------------------------------------------- */
// Defines
#if defined(__SDCC_PIC12F675)
	#define IronSW	GP1
	#define OnOff	GP3
	#define Led	GP4
	#define	Iron	GP5
#elif defined(__SDCC_PIC12F1822)
	#define IronSW	PORTAbits.RA1
	#define OnOff	PORTAbits.RA3
	#define Led	LATA4
	#define	Iron	LATA5
#endif

#define	setAD	0x08
#define	ironAD	0x00
#define	sec	5
/* Configuration bits: adapt to your setup and needs */
typedef unsigned int word;
#if defined(__SDCC_PIC12F675)
	word __at _CONFIG CONFIG = _INTRC_OSC_NOCLKOUT & _WDT_ON & _PWRTE_ON & _MCLRE_OFF & _BODEN_ON & _CPD_OFF & _CP_OFF;
#elif defined(__SDCC_PIC12F1822)
	word __at _CONFIG1 CONFIG1 = _FOSC_INTOSC & _WDTE_ON & _PWRTE_ON & _MCLRE_OFF & _CPD_OFF & _CP_OFF;
	word __at _CONFIG2 CONFIG2 = _WRT_OFF & _PLLEN_ON & _BORV_HI & _DEBUG_OFF & _LVP_OFF;
#endif
// Functions
void delay_s(unsigned char count);
unsigned char EERead(unsigned char addr);
unsigned char ADC_Get(unsigned char cfg);

// Variables
volatile typedef union {
	unsigned char SS;
	struct {
	  unsigned char :1;
	  unsigned char :1;
	  unsigned char :1;
	  unsigned char :1;
	  unsigned char :1;
	  unsigned char BLINK:1;
	  unsigned char CHECK:1;
	  unsigned char IRON:1;
	};
} __SS_bits_t;
__SS_bits_t SS_bits;
#define CHECK			SS_bits.CHECK
#define IRON			SS_bits.IRON
#define BLINK			SS_bits.BLINK
#define SS			SS_bits.SS

unsigned char i,j,k,timer=sec;
unsigned char setTemp=0,ironTemp=0;

typedef unsigned char eeprom;
#if defined(__SDCC_PIC12F675)
__code eeprom __at 0x2100 __EEPROM[] = {
0x30	// Sleep timer
};
#endif

void isr() __interrupt 0 {
	if (TMR1IF) {
		__asm__ ("CLRWDT");
		if (!timer--) {
			if (!OnOff)
				#ifdef	_NOBT
					IRON = 1;
				#else
					IRON ^= 1;
				#endif
			timer = sec;
		}
		CHECK = 1;
		BLINK = 1;
		TMR1IF = 0;
	}
}

void delay_ms(unsigned char count) {
unsigned char i,j;

	for (i=0;i<count;i++)
		for (j=0;j<100;j++) ;
	return;
}

unsigned char EERead(unsigned char adr) {
	EEADR=adr;
	RD=1;
	return EEDAT;
}

unsigned char ADC_Get(unsigned char cfg) {
	ADCON0 = cfg;
	delay_ms(10);
	ADON = 1;
	delay_ms(10);
	NOT_DONE = 1;
	while(NOT_DONE);
	ADON = 0;
	return ADRESH;
}

void main() {
/*
__asm
	banksel 1 ;Bank 1
	call 3FFh ;Get the cal value
	movwf OSCCAL ;Calibrate
	banksel 0 ;Bank 0
__endasm;
*/
	SS = 0;
	ADFM = 1;
#if defined(__SDCC_PIC12F675)
	GPIO = 0;
	CMCON = 0x07;	// Comparator off
	ANSEL = 0x55;	// GP0, GP2 analog, Fosc/16
	TRISIO = 0x0F;	// GP0-GP3 inputs GP
	WPU = 0x0A;	// GP1, GP3 week pull-up enable
	NOT_GPPU = 0; // Enable pull-ups
#elif defined(__SDCC_PIC12F1822)
	OSCCON = 0x68;	// 4-MHz IntOsc
	delay_ms(1);
	PORTA = 0;
	LATA = 0;
	TRISA = 0x0F;	// RA0-RA3 inputs
	ANSELA = 0x55;	// RA0, RA2 analog, Fosc/16
	WPUA = 0x0A;	// RA1, RA3 week pull-up enable
	NOT_WPUEN = 0; // Enable pull-ups
#endif
	ADCON0 = 0x00;	// Left justified result, chanel 0
	TMR1IE = 1;
	PEIE = 1;
	T1CON = 0x31; // Timer1 1:4 ON
	GIE = 1;
	while (1) {
		if (IronSW && IRON) {
			if (CHECK) {
				Iron = 0;
				setTemp = ADC_Get(setAD);
				ironTemp = ADC_Get(ironAD);
				CHECK = 0;
			}
			if (IRON) {
				if (ironTemp >= setTemp) {
					Iron = 0;
					Led = 1;
					CHECK = 1;
				} else {
					if (ironTemp < 0x30) { // Preheat
						Iron ^= 1;
						delay_ms(100);
					} else {
						Iron = 1;
					}
					if (BLINK) {
						Led ^= 1;
						BLINK = 0;
					}
				}
			}
		} else {
			Iron = 0;
			Led = 0;
		}
	}
}
