/* ----------------------------------------------------------------------- */
// Simple soldering station controller
// GP0 - Iron AD 
// GP1 - Iron Switch 
// GP2 - Iron Set 
// GP3 - On/Off 
// GP4 - Led 
// GP5 - Iron Heater 
/* ----------------------------------------------------------------------- */

#include <pic16regs.h>

/* ----------------------------------------------------------------------- */
// Defines
//#define	MAXTEMP	175 // 2*175 = 350 C
//#define	SLEEPTIME	180 // sec
#define	BTDELAY	5
#if defined(__SDCC_PIC12F675)
	#define IronSW	GP1
	#define OnOff	GP3
	#define Led	GP4
	#define	Iron	GP5
#elif defined(__SDCC_PIC12F1822)
	#define	GPIE	IOCIE
	#define	GPIF	IOCIF
	#define EEADR	EEADRL
	#define	EEDATA	EEDATL
	#define	IOC	IOCAF
	#define IronSW	PORTAbits.RA1
	#define OnOff	PORTAbits.RA3
	#define Led	LATA4
	#define	Iron	LATA5
#endif

#define	setAD	0x08
#define	ironAD	0x00
/* Configuration bits: adapt to your setup and needs */
typedef unsigned int word;
#if defined(__SDCC_PIC12F675)
	word __at _CONFIG CONFIG = _INTRC_OSC_NOCLKOUT & _WDT_ON & _PWRTE_ON & _MCLRE_OFF & _BODEN_ON & _CPD_OFF & _CP_OFF;
#elif defined(__SDCC_PIC12F1822)
	word __at _CONFIG1 CONFIG1 = _FOSC_INTOSC & _WDTE_ON & _PWRTE_ON & _MCLRE_OFF & _CPD_OFF & _CP_OFF;
	word __at _CONFIG2 CONFIG2 = _WRT_OFF & _PLLEN_ON & _BORV_HI & _DEBUG_OFF & _LVP_OFF;
#endif
// Functions
void delay_s(unsigned char count);
unsigned char EERead(unsigned char addr);
unsigned char ADC_Get(unsigned char cfg);

// Variables
volatile typedef union {
	unsigned char SS;
	struct {
	  unsigned char BUTTON:1;
	  unsigned char MINUS:1;
	  unsigned char PLUS:1;
	  unsigned char SAVE:1;
	  unsigned char SLEEP:1;
	  unsigned char BLINK:1;
	  unsigned char CHECK:1;
	  unsigned char IRON:1;
	};
} __SS_bits_t;
__SS_bits_t SS_bits;
#define BUTTON			SS_bits.BUTTON
#define MINUS			SS_bits.MINUS
#define PLUS			SS_bits.PLUS
#define SAVE			SS_bits.SAVE
#define SLEEP			SS_bits.SLEEP
#define CHECK			SS_bits.CHECK
#define IRON			SS_bits.IRON
#define BLINK			SS_bits.BLINK
#define SS			SS_bits.SS

unsigned char i,j,k,timer=BTDELAY,sleep=0,led_pwm=0,prog,btdelay=BTDELAY;
unsigned char eetemp,eeprog,eesleep,eemaxtemp;
unsigned char setTemp=0,ironTemp=0;

#if defined(__SDCC_PIC12F675)
	#define	__EEPROM_ADDR	0x2100
#else
	#define	__EEPROM_ADDR	0xF000
#endif

typedef unsigned char eeprom;
__code eeprom __at __EEPROM_ADDR __EEPROM[] = {
0x5A,	// 180 degrees
0x64, 	// 200 degrees
0x6E,	// 220 degrees
0x78,	// 240 degrees
0x00,
0x00, 0x00, 0x00, 0x00, 0x00,
0x00,	// Last profile used
0xB4,	// Sleep timer 180 seconds
0xAF,	// Max set temperature 350 degrees
};

void isr() __interrupt 0 {
	if (TMR1IF) {
		__asm__ ("CLRWDT");
		if (!btdelay) {
			if (!OnOff) {
				SLEEP = 0;
				BUTTON = 1;
				if (timer) timer--;
				else {
					IRON ^= 1;
					btdelay = timer = BTDELAY;
					BUTTON = 0;
					if (!IRON) SAVE = 1;

				}
			} else {
				if (BUTTON)
					if (timer)
						PLUS = 1;
				btdelay = timer = BTDELAY;
				BUTTON = 0;
			}
		} else btdelay--;
		#ifdef _SLEEP
			if (!sleep) {
				SLEEP = 1;
			} else {
				if (IRON)
					sleep--;
			}
		#endif
		CHECK = 1;
		BLINK = 1;
		TMR1IF = 0;
	}
	
	if (GPIF) {
		SLEEP = 0;
		sleep = eesleep;
		GPIF = 0;
	}
}

void delay_ms(unsigned char count) {
unsigned char i,j;

	for (i=0;i<count;i++)
		for (j=0;j<100;j++) ;
	return;
}

unsigned char EERead(unsigned char adr) {
	EEADR=adr;
	RD=1;
	return EEDATA;
}


void EEWrite(unsigned char val, unsigned char adr) {
	EEADR=adr;
	EEDATA=val;
	WREN=1;
	GIE=0;
	EECON2=0x55;
	EECON2=0xAA;
	WR=1;
	while (!EEIF);
	EEIF=0;
	GIE=1;
	WREN=0;
	return;
}

unsigned char ADC_Get(unsigned char cfg) {
	ADCON0 = cfg;
	delay_ms(10);
	ADON = 1;
	delay_ms(10);
	NOT_DONE = 1;
	while(NOT_DONE);
	ADON = 0;
	return ADRESH;
}

void LoadProg(unsigned char pnum) {
	setTemp = EERead(pnum);
	Iron = 0;
	Led = 0;
	delay_ms(0xFF);
	do {
		Led = 1;
		delay_ms(0x80);
		Led = 0;
		delay_ms(0x80);
	} while(pnum--);
	delay_ms(0xFF);
}

void main() {
/*
__asm
	banksel 1 ;Bank 1
	call 3FFh ;Get the cal value
	movwf OSCCAL ;Calibrate
	banksel 0 ;Bank 0
__endasm;
*/
	SS = 0;
	ADFM = 1;
#if defined(__SDCC_PIC12F675)
	GPIO = 0;
	CMCON = 0x07;	// Comparator off
	ANSEL = 0x55;	// GP0, GP2 analog, Fosc/16
	TRISIO = 0x0F;	// GP0-GP3 inputs GP
	WPU = 0x0A;	// GP1, GP3 week pull-up enable
	NOT_GPPU = 0; // Enable pull-ups
#elif defined(__SDCC_PIC12F1822)
	OSCCON = 0x68;	// 4-MHz IntOsc
	delay_ms(1);
	PORTA = 0;
	LATA = 0;
	TRISA = 0x0F;	// RA0-RA3 inputs
	ANSELA = 0x55;	// RA0, RA2 analog, Fosc/16
	WPUA = 0x0A;	// RA1, RA3 week pull-up enable
	NOT_WPUEN = 0; // Enable pull-ups
#endif
	ADCON0 = 0x00;	// Left justified result, chanel 0
	TMR1IE = 1;
	PEIE = 1;
	T1CON = 0x31; // Timer1 1:4 ON
#ifdef	_SLEEP
	IOC = 0x0A; // GP1, GP3 Interrupt-on-change enable
	GPIE = 1; // Eneble interrupt
#endif
// Load configuration for EEPROM
	prog = eeprog = EERead(0x0A);
	LoadProg(eeprog);
	sleep = eesleep = EERead(0x0B);
	eemaxtemp = EERead(0x0C);
	GIE = 1;
	#ifdef	_NOBT
		IRON = 1;
	#endif
	while (1) {
		if (SLEEP) {
			Iron = 0;
			i=0;
			while (i<led_pwm) {
				i++;
				Led = 0;
			}
			while (i) {
				i++;
				Led = 1;
			}
			led_pwm++;
		}
		else
			#ifdef	_SINGLE_BT
			if (PLUS) {
				if (++prog>3)
					prog = 0;
				LoadProg(prog);
				PLUS = 0;
			}
			#elif	_DUAL_BT
			#endif
			if (IronSW && IRON) {
				if (CHECK) {
					Iron = 0;
					#if !defined(_SINGLE_BT) && !defined(_DUAL_BT)
						setTemp = ADC_Get(setAD);
					#endif
					if (setTemp > eemaxtemp)
						setTemp = eemaxtemp;
					ironTemp = ADC_Get(ironAD);
					CHECK = 0;
				}
				if (IRON) {
					if (ironTemp >= setTemp) {
						Iron = 0;
						Led = 1;
						CHECK = 1;
					} else {
						if (ironTemp < 0x30) { // Preheat
							Iron ^= 1;
							delay_ms(100);
						} else {
							Iron = 1;
						}
						if (BLINK) {
							Led ^= 1;
							BLINK = 0;
						}
					}
				}
			} else {
				Iron = 0;
				Led = 0;
			}
			
		if (SAVE) {
			if (eeprog != prog)
				EEWrite(prog, 0x0A);
			SAVE = 0;
		}
	}
}
