

#include <string.h>
#include "n5110.h"

#define SET_SEND_CHR  	LL_GPIO_SetOutputPin(D_C_GPIO_Port, D_C_Pin)
#define SET_SEND_CMD	LL_GPIO_ResetOutputPin(D_C_GPIO_Port, D_C_Pin)

void SPI_DMA_TX (uint32_t MemoryAddress, uint16_t sz);

//    

static void LCDSend    ( uint8_t data, LcdCmdData cd );

//  

//    84*48   504 
 uint8_t  LcdCache [ LCD_CACHE_SIZE ];

//     ,      ,
//        . 
//          .
static int   LoWaterMark;   //  
static int   HiWaterMark;   //  

//     LcdCache[]
static int   LcdCacheIdx;

//   
static uint8_t  UpdateLcd;

//--------------------------------------------------------------------------------------------------

/*
 *                    :  Lcd_init
 *               :      SPI ,  LCD
 * ()           :  
 *   :  
 */

const uint8_t init_byte[]=
{
		0x21,   //     (LCD Extended Commands)
		0xB8,   //       6,42 
		0x04,   //    (Temp coefficent)
		0x13,   //   (LCD bias mode 1:48),0x14  biad mode 1:40
//0xC6,//   (LCD Vop)
		0x20,   //       
		0x0C //   (LCD in normal mode)
};

void LCDInit ( void )
{
	while(LL_SPI_IsActiveFlag_BSY(SPI1));

	//  reset
	LL_GPIO_ResetOutputPin(RST_GPIO_Port, RST_Pin);
	LL_mDelay(5);
	LL_GPIO_SetOutputPin(RST_GPIO_Port, RST_Pin);

	//   
	SET_SEND_CMD;
	SPI_DMA_TX((uint32_t) &init_byte, sizeof(init_byte));

	//   
	LCDClear();
}

//-----------------------------------------------------------------------------------------------------
/*
 *                    :  LCDClear
 *               :   .    LcdUpdate
 * ()           :  
 *   :  
 */
void LCDClear ( void )
{
    memset( LcdCache, 0x00, LCD_CACHE_SIZE );
    
    //      
    LoWaterMark = 0;
    HiWaterMark = LCD_CACHE_SIZE - 1;

    //    
    UpdateLcd = TRUE;
}
//-----------------------------------------------------------------------------------------------------
/*
 *                    :  LCDClearStr
 *               :   .    LcdUpdate
 * ()           :  y     ->  
 * 						 : 	count ->  
 *   :  
 */
void LCDClearStr(uint8_t y, uint8_t count)// 
{
//   .      504 
 LcdCacheIdx = y * 84;
 memset( &LcdCache[LcdCacheIdx], 0x00, 84 * count);
 LcdCacheIdx+=84 * count;
 if (LcdCacheIdx  >= LCD_CACHE_SIZE )
	 LcdCacheIdx = LCD_CACHE_SIZE - 1;

 if ( LcdCacheIdx > HiWaterMark )
  {
   //   
   HiWaterMark = LcdCacheIdx;
  }
 //    
   UpdateLcd = TRUE;
}

//------------------------------------------------------------------------------------------------------
/*
 *                    :  Lcd_update
 *               :      
 * ()           :  
 *   :  
 */
void Lcd_update (void)
{
    //uint16_t i;

    if(UpdateLcd == FALSE)return;

    if ( LoWaterMark < 0 )
        LoWaterMark = 0;
    else if ( LoWaterMark >= LCD_CACHE_SIZE )
        LoWaterMark = LCD_CACHE_SIZE - 1;

    if ( HiWaterMark < 0 )
        HiWaterMark = 0;
    else if ( HiWaterMark >= LCD_CACHE_SIZE )
        	HiWaterMark = LCD_CACHE_SIZE - 1;

	//  
    while(LL_SPI_IsActiveFlag_BSY(SPI1));//  
    LCDSend(0x40 | (LoWaterMark / LCD_X_RES), SEND_CMD);
	LCDSend(0x80 | (LoWaterMark % LCD_X_RES), SEND_CMD);

	//     
	while(LL_SPI_IsActiveFlag_BSY(SPI1));//  
	SET_SEND_CHR;// D/C  
	SPI_DMA_TX((uint32_t) &LcdCache[LoWaterMark], HiWaterMark - LoWaterMark + 1);

	//     
	LoWaterMark = LCD_CACHE_SIZE - 1;
	HiWaterMark = 0;

	//    
	UpdateLcd = FALSE;
}

//-------------------------------------------------------------------------------------------------------
/*
 *                    :  LCDSend
 *               :      
 * ()           :  data ->   
 *                          cd   ->    ( enum  n5110.h)
 *   :  
 */
static void LCDSend ( uint8_t data, LcdCmdData cd )
{
	// command or data
  if(cd == SEND_CHR)
   {
	  SET_SEND_CHR;// D/C  
   }
  else
   {
	  SET_SEND_CMD;// D/C  
   }

  while(!LL_SPI_IsActiveFlag_TXE(SPI1));
  LL_SPI_TransmitData8(SPI1, data);
}

//-----------------------------------------------------------------------------------------------------
/*
 *                    :  LcdContrast
 *               :    
 * ()           :   ->   0x00  0x7F
 *   :  
 */
void LcdContrast ( uint8_t contrast )
{
	 while(LL_SPI_IsActiveFlag_BSY(SPI1));
    LCDSend( 0x21, SEND_CMD );              //   
    LCDSend( 0x80 | contrast, SEND_CMD );   //   
    LCDSend( 0x20, SEND_CMD );              //   ,  
    //LL_mDelay(1);
}

//-----------------------------------------------------------------------------------------------------------
/*
 *                    :  LCDXY
 *               :      x,y    
 * ()           :  x,y ->    . : 0,0 .. 13,5
 *   :      n5110.h
 */
uint8_t LCD_XY ( uint8_t x, uint8_t y )
{
    //  
    if( x > 13 || y > 5 ) return OUT_OF_BORDER;

    //   .      504 
    LcdCacheIdx = x * 6 + y * 84;

    UpdateLcd = TRUE;

    return OK;
}

//-------------------------------------------------------------------------------------------------------

/*
 *                    :  LcdChr
 *               :       ,    
 * ()           :  size ->  .  enum  n5110.h
 *                          ch   ->   
 *   :      n5110lcd.h
 */

uint8_t LcdChr ( LcdFontSize size, uint8_t ch )
{
    uint8_t i, c;
    uint8_t b1, b2;
    int  tmpIdx;

    if ( LcdCacheIdx < LoWaterMark )
		{
			//   
			LoWaterMark = LcdCacheIdx;
		}

    if ( (ch >= 0x20) && (ch <= 0x7F) )
		{
			//      ASCII[0x20-0x7F]
			ch -= 32;
		}
    else if ( ch >= 0xC0 )
		{
			//      CP1251[0xC0-0xFF]
			ch -= 96;
		}
    else
		{
			//   (       )
			ch = 95;
		}

    if ( (size & 0x03) == FONT_1X )
		{
		  for ( i = 0; i < 5; i++ )
			{
				//       
				//LcdCache[LcdCacheIdx++] = pgm_read_uint8_t( &(FontLookup[ch][i]) ) << 1;
				c = FontLookup[ch][i];
				if((size & 0x04) == FONT_NEG )
									   c=~c;
				LcdCache[LcdCacheIdx++] = c;
			}
		}
    else if ( (size & 0x03) == FONT_2X )
		{
			tmpIdx = LcdCacheIdx - 84;

			if ( tmpIdx < LoWaterMark )
			{
				LoWaterMark = tmpIdx;
			}

			if ( tmpIdx < 0 ) return OUT_OF_BORDER;

			for ( i = 0; i < 5; i++ )
				{
					//        
					//c = pgm_read_uint8_t(&(FontLookup[ch][i])) << 1;
					c = FontLookup[ch][i];
					if((size & 0x04) == FONT_NEG ) c=~c;

					//  
					//  
					b1 =  (c & 0x01) * 3;
					b1 |= (c & 0x02) * 6;
					b1 |= (c & 0x04) * 12;
					b1 |= (c & 0x08) * 24;

					c >>= 4;
					//  
					b2 =  (c & 0x01) * 3;
					b2 |= (c & 0x02) * 6;
					b2 |= (c & 0x04) * 12;
					b2 |= (c & 0x08) * 24;

					//     
					LcdCache[tmpIdx++] = b1;
					LcdCache[tmpIdx++] = b1;
					LcdCache[tmpIdx + 82] = b2;
					LcdCache[tmpIdx + 83] = b2;
				}

			//  x  
			LcdCacheIdx = (LcdCacheIdx + 10) % LCD_CACHE_SIZE;
		}

    if ( LcdCacheIdx > HiWaterMark )
		{
			//   
			HiWaterMark = LcdCacheIdx;
		}

    //    
    c=((size & 0x04) == FONT_NEG ) ? 0xFF : 0x00;
    if ( (size & 0x03) == FONT_1X ) LcdCache[LcdCacheIdx] = c;
    else
		 {
			LcdCache[LcdCacheIdx-84] = c;
			LcdCache[LcdCacheIdx++] = c;
			LcdCache[LcdCacheIdx-84] = c;
			LcdCache[LcdCacheIdx] = c;
		 }

    //     LCD_CACHE_SIZE - 1,   
    if(LcdCacheIdx == (LCD_CACHE_SIZE - 1) )
		{
			LcdCacheIdx = 0;
			return OK_WITH_WRAP;
		}
    //    
    LcdCacheIdx++;
    return OK;
}

//--------------------------------------------------------------------------------------------------------
/*
 *                    :  LCDdStr
 *               :        
 * ()           :  size    ->  .  enum  n5110.h
 *                          dataPtr ->      
 *						 :  x,y -> 
 *   :      n5110lcd.h
 */

uint8_t LCDStr (LcdFontSize size, const uint8_t *dataPtr )
{
	uint8_t c;
    uint8_t response;
    for ( c = *dataPtr ; c; ++dataPtr, c = *dataPtr )
      {
        //  
        response = LcdChr( size, c );
        if(response == OUT_OF_BORDER)
            return OUT_OF_BORDER;
      }

    return OK;
}
//--------------------------------------------------------------------------------------
void Lcd_image(uint8_t x, uint8_t y, uint16_t RegAddr, uint16_t len)
{
	uint8_t buf[2];

	LcdCacheIdx = y*84+x;

	if ( LcdCacheIdx < LoWaterMark )
	    {
	        //   
	        LoWaterMark = LcdCacheIdx;
	    }

	buf[0]=(RegAddr>>8);buf[1]=(RegAddr);

	I2C_Master_Transmit(0xA0,buf, 2);

	if(LcdCacheIdx + len >= (LCD_CACHE_SIZE - 1))//     
		len = LCD_CACHE_SIZE - LcdCacheIdx-1;

	for(uint16_t i=0; i < len; i+=128)
	  {
	   I2C_Master_Receive(0xA1,&LcdCache[i],((len - i) > 128) ? 128 : len - i);
	  }

	LcdCacheIdx+=len;

	if ( LcdCacheIdx > HiWaterMark )
	  {
	    //   
	   HiWaterMark = LcdCacheIdx;
	  }

	UpdateLcd = TRUE;
}

//-------------------------------------------------------------------------------------------------------
/*
 *                    :  Lcd_pixel
 *               :       (x,y)
 * ()           :  x,y  ->   
 *                          mode -> Off, On  Xor.  enum  n5110.h
 *   :      n5110lcd.h
 */
uint8_t Lcd_pixel ( uint8_t x, uint8_t y, LcdPixelMode mode )
{
    int  index;
    uint8_t  offset;
    uint8_t  data;

    //     
    if ( x >= LCD_X_RES || y >= LCD_Y_RES) return OUT_OF_BORDER;

    //    
    index = ( ( y / 8 ) * 84 ) + x;
    offset  = y - ( ( y / 8 ) * 8 );

    data = LcdCache[ index ];

    //  

    //  PIXEL_OFF
    if ( mode == PIXEL_OFF )
    {
        data &= ( ~( 0x01 << offset ) );
    }
    //  PIXEL_ON
    else if ( mode == PIXEL_ON )
    {
        data |= ( 0x01 << offset );
    }
    //  PIXEL_XOR
    else if ( mode  == PIXEL_XOR )
    {
        data ^= ( 0x01 << offset );
    }

    //     
    LcdCache[ index ] = data;

    if ( index < LoWaterMark )
    {
        //   
        LoWaterMark = index;
    }

    if ( index > HiWaterMark )
    {
        //   
        HiWaterMark = index;
    }
    return OK;
}
//------------------------------------------------------------------------------------------------------
/*
 *                    :  Lcd_line
 *               :         ( )
 * ()           :  x1, y1  ->    
 *                          x2, y2  ->    
 *                          mode    -> Off, On  Xor.  enum  n5110.h
 *   :      n5110lcd.h
 */
uint8_t Lcd_line ( uint8_t x1, uint8_t y1, uint8_t x2, uint8_t y2, LcdPixelMode mode )
{
    int dx, dy, stepx, stepy, fraction;
    uint8_t response;

    // dy   y2 - y1
    // -- = -------
    // dx   x2 - x1

    dy = y2 - y1;
    dx = x2 - x1;

    // dy 
    if ( dy < 0 )
    {
        dy    = -dy;
        stepy = -1;
    }
    else
    {
        stepy = 1;
    }

    // dx 
    if ( dx < 0 )
    {
        dx    = -dx;
        stepx = -1;
    }
    else
    {
        stepx = 1;
    }

    dx <<= 1;
    dy <<= 1;

    //   
    response = Lcd_pixel( x1, y1, mode );
    if(response)
        return response;

    //     
    if ( dx > dy )
    {
        fraction = dy - ( dx >> 1);
        while ( x1 != x2 )
        {
            if ( fraction >= 0 )
            {
                y1 += stepy;
                fraction -= dx;
            }
            x1 += stepx;
            fraction += dy;

            response = Lcd_pixel( x1, y1, mode );
            if(response)
                return response;

        }
    }
    else
    {
        fraction = dx - ( dy >> 1);
        while ( y1 != y2 )
        {
            if ( fraction >= 0 )
            {
                x1 += stepx;
                fraction -= dy;
            }
            y1 += stepy;
            fraction += dx;

            response = Lcd_pixel( x1, y1, mode );
            if(response)
                return response;
        }
    }

    //    
    UpdateLcd = TRUE;
    return OK;
}
//------------------------------------------------------------------------------------------------------
/*
 *                    :  Lcd_circle
 *               :    ( )
 * ()           :  x, y   ->   
 *                          radius ->  
 *                          mode   -> Off, On  Xor.  enum  n5110.h
 *   :      n5110lcd.h
 */
uint8_t Lcd_circle(uint8_t x, uint8_t y, uint8_t radius, LcdPixelMode mode)
{
    signed char xc = 0;
    signed char yc = 0;
    signed char p = 0;

    if ( x >= LCD_X_RES || y >= LCD_Y_RES) return OUT_OF_BORDER;

    yc = radius;
    p = 3 - (radius<<1);
    while (xc <= yc)  
    {
        Lcd_pixel(x + xc, y + yc, mode);
        Lcd_pixel(x + xc, y - yc, mode);
        Lcd_pixel(x - xc, y + yc, mode);
        Lcd_pixel(x - xc, y - yc, mode);
        Lcd_pixel(x + yc, y + xc, mode);
        Lcd_pixel(x + yc, y - xc, mode);
        Lcd_pixel(x - yc, y + xc, mode);
        Lcd_pixel(x - yc, y - xc, mode);
        if (p < 0) p += (xc++ << 2) + 6;
            else p += ((xc++ - yc--)<<2) + 10;
    }

    //    
    UpdateLcd = TRUE;
    return OK;
}

//-------------------------------------------------------------------------------------------------------
/*
 *                    :  Lcd_rect  (rectangle)
 *               :     
 * ()           :  baseX  ->   x (  )
 *                          baseY  ->   y (  )
 *                          height ->  ( )
 *                          width  ->  ( )
 *                          mode   -> Off, On  Xor.  enum  n5110.h
 *   :      n5110lcd.h
 */
uint8_t Lcd_rect ( uint8_t baseX, uint8_t baseY, uint8_t height, uint8_t width, LcdPixelMode mode )
{
    uint8_t tmpIdxX,tmpIdxY,tmp;

    uint8_t response;

    //  
    if ( ( baseX >= LCD_X_RES) || ( baseY >= LCD_Y_RES) ) return OUT_OF_BORDER;

    if ( height > baseY )
        tmp = 0;
    else
        tmp = baseY - height + 1;

    //  
    for ( tmpIdxY = tmp; tmpIdxY <= baseY; tmpIdxY++ )
    {
        for ( tmpIdxX = baseX; tmpIdxX < (baseX + width); tmpIdxX++ )
        {
            response = Lcd_pixel( tmpIdxX, tmpIdxY, mode );
            if(response)
                return response;

        }
    }

    //    
    UpdateLcd = TRUE;
    return OK;
}
//----------------------------------------------------------------------------------------------
/*
 *                    :  Lcd_rect_empty
 *               :    
 * ()           :  x1    ->   x   
 *                          y1    ->   y   
 *                          x2    ->   x   
 *                          y2    ->   y   
 *                          mode  -> Off, On  Xor.  enum  n5110.h
 *   :      n5110lcd.h
 */
uint8_t Lcd_rect_empty ( uint8_t x1, uint8_t y1, uint8_t x2, uint8_t y2, LcdPixelMode mode )
{
    uint8_t tmpIdx;

    //  
    if ( ( x1 >= LCD_X_RES) ||  ( x2 >= LCD_X_RES) || ( y1 >= LCD_Y_RES) || ( y2 >= LCD_Y_RES) )
        return OUT_OF_BORDER;

    if ( ( x2 > x1 ) && ( y2 > y1 ) )
    {
        //   
        for ( tmpIdx = x1; tmpIdx <= x2; tmpIdx++ )
        {
            Lcd_pixel( tmpIdx, y1, mode );
            Lcd_pixel( tmpIdx, y2, mode );
        }

        //   
        for ( tmpIdx = y1; tmpIdx <= y2; tmpIdx++ )
        {
            Lcd_pixel( x1, tmpIdx, mode );
            Lcd_pixel( x2, tmpIdx, mode );
        }

        //    
        UpdateLcd = TRUE;
    }
    return OK;
}

